<?php
/**
 * Admin class for settings page
 */

if (!defined('ABSPATH')) {
    exit;
}

class WP_GA4_SST_Admin {
    
    /**
     * Constructor
     */
    public function __construct() {
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_init', array($this, 'register_settings'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_styles'));
        add_filter('plugin_action_links_' . plugin_basename(WP_GA4_SST_PLUGIN_DIR . 'wp-ga4-sst.php'), array($this, 'add_settings_link'));
    }
    
    /**
     * Admin-Styles load
     */
    public function enqueue_admin_styles($hook) {
        // Load styles only on our plugin pages
        if (!in_array($hook, array('toplevel_page_wp-ga4-sst-main', 'ga4-sst_page_wp-ga4-sst-logs'))) {
            return;
        }
        
        wp_enqueue_style('wp-ga4-sst-colors', WP_GA4_SST_PLUGIN_URL . 'css/colors.css', array(), WP_GA4_SST_VERSION);
        wp_enqueue_style('wp-ga4-sst-admin', WP_GA4_SST_PLUGIN_URL . 'css/admin.css', array('wp-ga4-sst-colors'), WP_GA4_SST_VERSION);
        
        // Load logs CSS on logs page
        if ($hook === 'ga4-sst_page_wp-ga4-sst-logs') {
            wp_enqueue_style('wp-ga4-sst-logs', WP_GA4_SST_PLUGIN_URL . 'css/logs.css', array('wp-ga4-sst-colors'), WP_GA4_SST_VERSION);
        }
    }
    
    /**
     * Admin-Menü add
     */
    public function add_admin_menu() {
        // Add main menu
        add_menu_page(
            __('GA4 Server-Side Tracking', 'seoday-wp-ga4-sst'),
            __('GA4-SST', 'seoday-wp-ga4-sst'),
            'manage_options',
            'wp-ga4-sst-main',
            array($this, 'render_settings_page'),
            'dashicons-chart-line',
            30
        );
        
        // Add Logs submenu
        add_submenu_page(
            'wp-ga4-sst-main',
            __('GA4 Logs', 'seoday-wp-ga4-sst'),
            __('GA4-Logs', 'seoday-wp-ga4-sst'),
            'manage_options',
            'wp-ga4-sst-logs',
            array($this, 'render_logs_page')
        );
    }
    
    /**
     * Register settings
     */
    public function register_settings() {
        // Register settings
        register_setting('wp_ga4_sst_settings', 'wp_ga4_sst_enabled', array(
            'type' => 'boolean',
            'default' => false,
            'sanitize_callback' => array($this, 'sanitize_checkbox')
        ));
        
        register_setting('wp_ga4_sst_settings', 'wp_ga4_sst_measurement_id', array(
            'type' => 'string',
            'sanitize_callback' => 'sanitize_text_field'
        ));
        
        register_setting('wp_ga4_sst_settings', 'wp_ga4_sst_api_secret', array(
            'type' => 'string',
            'sanitize_callback' => 'sanitize_text_field'
        ));
        
        // Add section
        add_settings_section(
            'wp_ga4_sst_main_section',
            __('Configuration', 'seoday-wp-ga4-sst'),
            array($this, 'render_section_info'),
            'seoday-wp-ga4-sst'
        );
        
        // Add fields
        add_settings_field(
            'wp_ga4_sst_enabled',
            __('Enable Tracking', 'seoday-wp-ga4-sst'),
            array($this, 'render_enabled_field'),
            'seoday-wp-ga4-sst',
            'wp_ga4_sst_main_section'
        );
        
        add_settings_field(
            'wp_ga4_sst_measurement_id',
            __('Measurement ID', 'seoday-wp-ga4-sst'),
            array($this, 'render_measurement_id_field'),
            'seoday-wp-ga4-sst',
            'wp_ga4_sst_main_section'
        );
        
        add_settings_field(
            'wp_ga4_sst_api_secret',
            __('API Secret', 'seoday-wp-ga4-sst'),
            array($this, 'render_api_secret_field'),
            'seoday-wp-ga4-sst',
            'wp_ga4_sst_main_section'
        );
    }
    
    /**
     * Checkbox-Wert sanitize
     */
    public function sanitize_checkbox($input) {
        return (bool) $input;
    }
    
    /**
     * Sectionsbeschreibung render
     */
    public function render_section_info() {
        include WP_GA4_SST_PLUGIN_DIR . 'templates/section-info.php';
    }
    
    /**
     * Tracking aktivieren Field render
     */
    public function render_enabled_field() {
        $enabled = get_option('wp_ga4_sst_enabled', false);
        $measurement_id = get_option('wp_ga4_sst_measurement_id', '');
        $api_secret = get_option('wp_ga4_sst_api_secret', '');
        
        // Check if both measurement ID and API secret are set
        $can_activate = !empty($measurement_id) && !empty($api_secret);
        
        include WP_GA4_SST_PLUGIN_DIR . 'templates/field-enabled.php';
    }
    
    /**
     * Measurement ID Field render
     */
    public function render_measurement_id_field() {
        $measurement_id = get_option('wp_ga4_sst_measurement_id', '');
        include WP_GA4_SST_PLUGIN_DIR . 'templates/field-measurement-id.php';
    }
    
    /**
     * API Secret Field render
     */
    public function render_api_secret_field() {
        $api_secret = get_option('wp_ga4_sst_api_secret', '');
        include WP_GA4_SST_PLUGIN_DIR . 'templates/field-api-secret.php';
    }
    
    /**
     * Einstellungsseite render
     */
    public function render_settings_page() {
        // Check permissions
        if (!current_user_can('manage_options')) {
            return;
        }
        
        // Show success message
        if (isset($_GET['settings-updated'])) {
            add_settings_error(
                'wp_ga4_sst_messages',
                'wp_ga4_sst_message',
                __('Settings saved.', 'seoday-wp-ga4-sst'),
                'updated'
            );
        }
        
        settings_errors('wp_ga4_sst_messages');
        
        // Prepare status table
        ob_start();
        $this->render_status();
        $status_table = ob_get_clean();
        
        // Load template
        include WP_GA4_SST_PLUGIN_DIR . 'templates/admin-settings.php';
    }
    
    /**
     * Status render
     */
    private function render_status() {
        $enabled = get_option('wp_ga4_sst_enabled', false);
        $measurement_id = get_option('wp_ga4_sst_measurement_id', '');
        $api_secret = get_option('wp_ga4_sst_api_secret', '');
        
        include WP_GA4_SST_PLUGIN_DIR . 'templates/status-table.php';
    }
    
    /**
     * Logs-Seite render
     */
    public function render_logs_page() {
        // Check permissions
        if (!current_user_can('manage_options')) {
            return;
        }
        
        // Load logs class
        if (!class_exists('WP_GA4_SST_Logs')) {
            require_once WP_GA4_SST_PLUGIN_DIR . 'includes/class-wp-ga4-sst-logs.php';
        }
        
        $logs_handler = new WP_GA4_SST_Logs();
        $logs_handler->render_logs_page();
    }
    
    /**
     * Einstellungs-Link zu Plugin-Aktionen add
     */
    public function add_settings_link($links) {
        $settings_link = '<a href="' . admin_url('admin.php?page=wp-ga4-sst-main') . '">' . __('Settings', 'seoday-wp-ga4-sst') . '</a>';
        array_unshift($links, $settings_link);
        return $links;
    }
}

