<?php
/**
 * GA4 Server-Side Tracker Class
 */

if (!defined('ABSPATH')) {
    exit;
}

class WP_GA4_SST_Tracker {
    
    /**
     * GA4 Measurement Protocol Endpoint
     */
    private const GA4_ENDPOINT = 'https://www.google-analytics.com/mp/collect';
    
    /**
     * @var WP_GA4_SST_Settings
     */
    private $settings;
    
    /**
     * @var WP_GA4_SST_Call_Log_Repository
     */
    private $repository;
    
    /**
     * Client ID
     */
    private $client_id;
    
    /**
     * Constructor
     */
    public function __construct() {
        $this->settings = WP_GA4_SST_Settings::load_from_options();
        $this->repository = new WP_GA4_SST_Call_Log_Repository();
        
        // Only initialize if configured
        if ($this->settings->has_credentials()) {
            $this->init_hooks();
        }
    }
    
    /**
     * Hooks initialize
     */
    private function init_hooks() {
        add_action('wp', array($this, 'track_all_events'));
    }
    
    /**
     * Check if this is a first visit
     */
    private function is_first_visit() {
        $cookie_name = 'technical_website_cookie';
        return !isset($_COOKIE[$cookie_name]);
    }
    
    /**
     * Check if this is a new session
     */
    private function is_new_session() {
        $session_cookie = 'technical_website_session';
        return !isset($_COOKIE[$session_cookie]);
    }
    
    /**
     * Mark session as started
     */
    private function mark_session_started() {
        $session_cookie = 'technical_website_session';
        // Session cookie expires after 30 minutes of inactivity
        if (!headers_sent()) {
            setcookie($session_cookie, '1', time() + (30 * 60), '/');
        }
    }
    
    /**
     * Generate or get Client ID
     */
    private function get_client_id() {
        if ($this->client_id) {
            return $this->client_id;
        }
        
        // Use cookie for Client ID
        $cookie_name = 'technical_website_cookie';
        
        if (isset($_COOKIE[$cookie_name])) {
            $this->client_id = sanitize_text_field($_COOKIE[$cookie_name]);
        } else {
            // Generate new Client ID
            $this->client_id = $this->generate_client_id();
            
            // Set cookie (permanent - 10 years)
            if (!headers_sent()) {
                setcookie($cookie_name, $this->client_id, time() + (365 * 10 * 24 * 60 * 60), '/');
            }
        }
        
        return $this->client_id;
    }
    
    /**
     * Generate Client ID
     */
    private function generate_client_id() {
        return sprintf('%d.%d', time(), wp_rand(100000000, 999999999));
    }
    
    /**
     * Track all events in a single call
     */
    public function track_all_events() {
        // Don't track in admin area
        if (is_admin()) {
            return;
        }
        
        $events = array();
        
        // Check for first visit
        $is_first_visit = $this->is_first_visit();
        
        // Check for new session
        $is_new_session = $this->is_new_session();
        
        // Add session_start event if needed
        if ($is_new_session) {
            $params = new WP_GA4_SST_Event_Params();
            $params->add_param('page_location', $this->get_page_url());
            $params->add_param('session_id', $this->get_client_id());
            $params->add_param('engagement_time_msec', 100);
            
            $events[] = new WP_GA4_SST_Event('session_start', $params);
            
            // Mark session as started
            $this->mark_session_started();
        }
        
        // Add first_visit event if needed
        if ($is_first_visit) {
            $params = new WP_GA4_SST_Event_Params();
            $params->add_param('page_location', $this->get_page_url());
            $params->add_param('engagement_time_msec', 100);
            
            $events[] = new WP_GA4_SST_Event('first_visit', $params);
        }
        
        // Add page_view event (always)
        $params = WP_GA4_SST_Event_Params::create_page_view(
            $this->get_page_url(),
            get_the_title() ?: wp_get_document_title(),
            isset($_SERVER['HTTP_REFERER']) ? esc_url_raw($_SERVER['HTTP_REFERER']) : '',
            get_locale()
        );
        $events[] = new WP_GA4_SST_Event('page_view', $params);
        
        // Send all events in a single call
        $this->send_events($events);
    }
    
    /**
     * Send multiple events to GA4 in a single call
     * 
     * @param array $events Array of WP_GA4_SST_Event objects
     */
    private function send_events($events) {
        if (empty($events)) {
            return;
        }
        
        $url = add_query_arg(
            array(
                'measurement_id' => $this->settings->get_measurement_id(),
                'api_secret' => $this->settings->get_api_secret(),
            ),
            self::GA4_ENDPOINT
        );
        
        // Convert events to array format
        $events_array = array();
        foreach ($events as $event) {
            $events_array[] = $event->to_array();
        }
        
        $payload = array(
            'client_id' => $this->get_client_id(),
            'events' => $events_array,
            'user_properties' => array(
                'user_role' => array(
                    'value' => $this->get_user_role()
                )
            )
        );
        
        // Session-Informationen add
        if (session_id()) {
            $payload['session_id'] = session_id();
        }
        
        // User ID add (falls eingeloggt)
        if (is_user_logged_in()) {
            $payload['user_id'] = (string) get_current_user_id();
        }
        
        $payload_json = wp_json_encode($payload);
        
        // Request senden
        $headers = array(
            'Content-Type' => 'application/json',
        );
        
        // Add User-Agent header if available
        if (isset($_SERVER['HTTP_USER_AGENT'])) {
            $headers['User-Agent'] = sanitize_text_field($_SERVER['HTTP_USER_AGENT']);
        }
        
        $response = wp_remote_post($url, array(
            'headers' => $headers,
            'body' => $payload_json,
            'timeout' => 5,
            'blocking' => true, // Blocking for logging
        ));
        
        // Extract response body
        $response_body = '';
        $response_code = 0;
        
        if (is_wp_error($response)) {
            $response_code = 0;
            $response_body = wp_json_encode(array(
                'error' => $response->get_error_message()
            ));
            
            // Error handling (only when debugging)
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('WP GA4 SST Error: ' . $response->get_error_message());
            }
        } else {
            $response_code = wp_remote_retrieve_response_code($response);
            $response_body = wp_remote_retrieve_body($response);
            
            if (empty($response_body)) {
                $response_body = wp_json_encode(array(
                    'status_code' => $response_code,
                    'message' => 'Success'
                ));
            }
        }
        
        // Save call to database
        $this->log_call($payload_json, $response_body, $response_code);
    }
    
    /**
     * Log GA4 call to database
     */
    private function log_call($request, $response, $response_code) {
        $domain = isset($_SERVER['HTTP_HOST']) ? sanitize_text_field($_SERVER['HTTP_HOST']) : '';
        
        // Create call log object
        $log = new WP_GA4_SST_Call_Log(
            current_time('mysql'),
            current_time('Y-m-d'),
            $domain,
            $response_code,
            $request,
            $response
        );
        
        // Save to database
        $this->repository->save($log);
        
        // Delete old entries (older than 30 days)
        $this->repository->delete_older_than(30);
    }
    
    /**
     * Get current page URL
     */
    private function get_page_url() {
        global $wp;
        return home_url(add_query_arg(array(), $wp->request));
    }
    
    /**
     * Get user role
     */
    private function get_user_role() {
        if (!is_user_logged_in()) {
            return 'visitor';
        }
        
        $user = wp_get_current_user();
        $roles = $user->roles;
        
        return !empty($roles) ? $roles[0] : 'subscriber';
    }
    
    /**
     * Send custom event (for developers)
     * 
     * @param string $event_name
     * @param array $params
     */
    public function track_event($event_name, $params = array()) {
        $event_params = new WP_GA4_SST_Event_Params($params);
        $event = new WP_GA4_SST_Event(sanitize_text_field($event_name), $event_params);
        
        // Send as single event in array
        $this->send_events(array($event));
    }
}

