<?php
/**
 * Repository: GA4 Call Log
 * Manages database accesse für Call Logs
 */

if (!defined('ABSPATH')) {
    exit;
}

class WP_GA4_SST_Call_Log_Repository {
    
    /**
     * @var wpdb
     */
    private $wpdb;
    
    /**
     * @var string
     */
    private $table_name;
    
    /**
     * Constructor
     */
    public function __construct() {
        global $wpdb;
        $this->wpdb = $wpdb;
        $this->table_name = $wpdb->prefix . 'ga4_sst_calls';
    }
    
    /**
     * Call Log save
     * 
     * @param WP_GA4_SST_Call_Log $log
     * @return bool
     */
    public function save(WP_GA4_SST_Call_Log $log) {
        $result = $this->wpdb->insert(
            $this->table_name,
            array(
                'timestamp' => $log->get_timestamp(),
                'day' => $log->get_day(),
                'domain' => $log->get_domain(),
                'response_code' => $log->get_response_code(),
                'request' => $log->get_request(),
                'response' => $log->get_response(),
            ),
            array('%s', '%s', '%s', '%d', '%s', '%s')
        );
        
        if ($result) {
            $log->set_id($this->wpdb->insert_id);
            return true;
        }
        
        return false;
    }
    
    /**
     * All Logs get mit Pagination
     * 
     * @param int $limit
     * @param int $offset
     * @param string|null $filter_day
     * @return WP_GA4_SST_Call_Log[]
     */
    public function get_all($limit = 20, $offset = 0, $filter_day = null) {
        $where = '';
        
        if ($filter_day) {
            $where = $this->wpdb->prepare(" WHERE day = %s", $filter_day);
        }
        
        $sql = $this->wpdb->prepare(
            "SELECT * FROM {$this->table_name}{$where} ORDER BY timestamp DESC LIMIT %d OFFSET %d",
            $limit,
            $offset
        );
        
        $results = $this->wpdb->get_results($sql);
        
        $logs = array();
        foreach ($results as $row) {
            $logs[] = WP_GA4_SST_Call_Log::from_db_row($row);
        }
        
        return $logs;
    }
    
    /**
     * Count aller Logs determine
     * 
     * @param string|null $filter_day
     * @return int
     */
    public function count($filter_day = null) {
        $where = '';
        
        if ($filter_day) {
            $where = $this->wpdb->prepare(" WHERE day = %s", $filter_day);
        }
        
        $sql = "SELECT COUNT(*) FROM {$this->table_name}{$where}";
        return (int) $this->wpdb->get_var($sql);
    }
    
    /**
     * Statistiken get
     * 
     * @return WP_GA4_SST_Log_Statistics
     */
    public function get_statistics() {
        $sql = "
            SELECT 
                COUNT(*) as total_calls,
                COUNT(DISTINCT day) as total_days,
                COUNT(DISTINCT domain) as total_domains
            FROM {$this->table_name}
        ";
        
        $row = $this->wpdb->get_row($sql);
        return WP_GA4_SST_Log_Statistics::from_db_row($row);
    }
    
    /**
     * Old Logs delete
     * 
     * @param int $days Number of days
     * @return int Count deleted entries
     */
    public function delete_older_than($days = 30) {
        $sql = $this->wpdb->prepare(
            "DELETE FROM {$this->table_name} WHERE day < DATE_SUB(CURDATE(), INTERVAL %d DAY)",
            $days
        );
        
        return $this->wpdb->query($sql);
    }
    
    /**
     * table create
     */
    public function create_table() {
        $charset_collate = $this->wpdb->get_charset_collate();
        
        $sql = "CREATE TABLE IF NOT EXISTS {$this->table_name} (
            id bigint(20) UNSIGNED NOT NULL AUTO_INCREMENT,
            timestamp datetime DEFAULT CURRENT_TIMESTAMP NOT NULL,
            day date NOT NULL,
            domain varchar(255) NOT NULL,
            response_code int(3) NOT NULL DEFAULT 0,
            request longtext NOT NULL,
            response longtext NOT NULL,
            PRIMARY KEY  (id),
            KEY day (day),
            KEY domain (domain),
            KEY timestamp (timestamp),
            KEY response_code (response_code)
        ) {$charset_collate};";
        
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
    }
    
    /**
     * Delete all logs
     */
    public function delete_all() {
        $this->wpdb->query("TRUNCATE TABLE {$this->table_name}");
    }
    
    /**
     * table delete
     */
    public function drop_table() {
        $this->wpdb->query("DROP TABLE IF EXISTS {$this->table_name}");
    }
}

